//Conventions:
//    Global constants (declared with const) and #defines - all uppercase letters with words separated 
//        by underscores.
//        (E.G., #define MY_DEFINE 5).
//        (E.G., const int MY_CONSTANT = 5;).
//    New data types (classes, structs, typedefs, etc.) - begin with an uppercase letter followed by
//        lowercase words separated by uppercase letters.  Enumerated constants contain a prefix
//        associating them with a particular enumerated set.
//        (E.G., typedef int MyTypedef;).
//        (E.G., enum MyEnumConst {MEC_ONE, MEC_TWO};)
//    Global variables - begin with "g_" followed by lowercase words separated by underscores.
//        (E.G., int g_my_global;).
//    Local variables - begin with a lowercase letter followed by lowercase words separated by
//        underscores.
//        (E.G., int my_local;).
//    Argument variables - begin with "a_" followed by lowercase words separated by underscores.
//        (E.G., ...int a_my_argument, ...).
//    Member variables - begin with "m_" followed by lowercase words separated by underscores.
//        (E.G., int m_my_member;).
//    Functions (member or global) - begin with an uppercase letter followed by lowercase words
//        separated by uppercase letters.
//        (E.G., void MyFunction(void);).


//******************************************************************************************************
//**** SYSTEM HEADER FILES
//******************************************************************************************************
#include <stdio.h>
#include <string.h>
#include <FilePanel.h>
#include <Path.h>
#include <MenuBar.h>
#include <MenuItem.h>
#include <NodeInfo.h>


//******************************************************************************************************
//**** PROJECT HEADER FILES
//******************************************************************************************************
#include "DocumentApp.h"
#include "FileDocumentWindow.h"
#include "WindowPositionSet.h"
#include "WrappingTextView.h"


//******************************************************************************************************
//**** CLASS DECLARATIONS
//******************************************************************************************************
class GenericApp : public DocumentApp
{
	public:
		GenericApp();
		virtual ~GenericApp();
		virtual void MessageReceived(BMessage* a_message);
     	virtual void RefsReceived(BMessage *a_message);
		virtual void ArgvReceived(int32 argc, char **argv);
		virtual void ReadyToRun();

	private:
		void OpenNewDocument();
		int32 m_doc_title_number;
		WindowPositionSet m_window_position_set;
		BFilePanel* m_open_panel;
};


class GenericTextView;
class GenericWindow : public FileDocumentWindow
{
	public:
		GenericWindow(BRect a_frame,const char* a_filename);
		virtual ~GenericWindow();
		virtual bool Save();

	private:
		BMenuBar* GenerateMenuBar();
		GenericTextView* m_text_view;
};


class GenericTextView : public WrappingTextView
{
	public:
		GenericTextView(BRect a_frame,const char* a_name,int32 a_resize_mode,int32 a_flags);
	virtual void Modified();
};


//******************************************************************************************************
//**** CONSTANTS
//******************************************************************************************************
const int32 MSG_NEW_DOCUMENT = 'NewD';	//LOL
const int32 MSG_OPEN_DOCUMENT = 'Open';


//******************************************************************************************************
//**** APPLICATION
//******************************************************************************************************
int main()
{
	new GenericApp;
	be_app->Run();
	delete be_app;
}


GenericApp::GenericApp()
: DocumentApp("application/x-vnd.BT-Generic")
{
	m_doc_title_number = 1;

	//Set up the open file panel
	m_open_panel = NULL;
	status_t error;
	BMessenger app_messenger((BHandler*)NULL,this,&error);
	if(error == B_OK)
		m_open_panel = new BFilePanel(B_OPEN_PANEL,&app_messenger,NULL);
}


GenericApp::~GenericApp()
{ }



void GenericApp::OpenNewDocument()
{
	char name_buffer[13];
	sprintf(name_buffer,"Untitled %d",(int)atomic_add(&m_doc_title_number,1));
	new GenericWindow(m_window_position_set.GetNextUntitledWindow(),name_buffer);
}


void GenericApp::MessageReceived(BMessage* a_message)
{
	switch(a_message->what)
	{
		case MSG_NEW_DOCUMENT:
			OpenNewDocument();
			break;
		case MSG_OPEN_DOCUMENT:
			m_open_panel->Show();
			break;
		default:
			DocumentApp::MessageReceived(a_message);
			break;
	}
}


void GenericApp::RefsReceived(BMessage* a_message)
{
	type_code type;
	int32 count;
	entry_ref file_entry_ref;

	a_message->GetInfo("refs", &type, &count);
	if(type != B_REF_TYPE)
		return;
	for(int32 ref_index = 0; ref_index < count; ref_index++)
	{
		if(a_message->FindRef("refs", ref_index, &file_entry_ref) == B_OK)
		{
			BEntry file_entry(&file_entry_ref,true);
			if(file_entry.InitCheck() == B_NO_ERROR)
			{
				BPath file_path;
				if(file_entry.GetPath(&file_path) == B_NO_ERROR)
					new GenericWindow(m_window_position_set.GetNextUntitledWindow(),file_path.Path());
			}
		}
	}
}


void GenericApp::ArgvReceived(int32 argc, char **argv)
{
	if(argc > 1)
		for(int32 counter = 1; counter < argc; counter++)
			new GenericWindow(m_window_position_set.GetNextUntitledWindow(),argv[counter]);
}


void GenericApp::ReadyToRun()
{
	if(CountRegWindProcHolds() == 0)
		//Start with a single window
		OpenNewDocument();
}


//******************************************************************************************************
//**** WINDOW
//******************************************************************************************************
GenericWindow::GenericWindow(BRect a_frame,const char* a_filename)
: FileDocumentWindow(a_frame,a_filename,B_DOCUMENT_WINDOW,B_WILL_ACCEPT_FIRST_CLICK)
{
	BMenuBar* menu_bar = GenerateMenuBar();
	AddChild(menu_bar);

	BRect r = Bounds();
	r.top = menu_bar->Frame().bottom+1.0;
	m_text_view = new GenericTextView(r,"Text view",B_FOLLOW_ALL_SIDES,B_WILL_DRAW);
	if(strncmp(a_filename,"Untitled",8) != 0)
	{
		BFile file(a_filename,B_READ_ONLY);
		off_t size;
		if(file.InitCheck() == B_NO_ERROR)
			if(file.GetSize(&size) == B_NO_ERROR)
				m_text_view->SetText(&file,0,size);
	}
	AddChild(m_text_view);

	Show();
}


BMenuBar* GenericWindow::GenerateMenuBar()
{
	BMenuBar* menu_bar = new BMenuBar(BRect(0,0,0,0),"Menu");
	BMenu* menu;
	BMenuItem* item;

	//File Menu
	menu = new BMenu("File");
		//New
			item = new BMenuItem("New document",new BMessage(MSG_NEW_DOCUMENT),'N');
			item->SetTarget(be_app);
			menu->AddItem(item);
		//Open...
			item = new BMenuItem("Open"B_UTF8_ELLIPSIS,new BMessage(MSG_OPEN_DOCUMENT),'O');
			item->SetTarget(be_app);
			menu->AddItem(item);
		menu->AddSeparatorItem();
		//Save
			menu->AddItem(new BMenuItem("Save",new BMessage(B_SAVE_REQUESTED),'S'));
		//Save as...
			menu->AddItem(new BMenuItem("Save as"B_UTF8_ELLIPSIS,new BMessage(FDW_SAVE_AS_REQESTED),0));
		menu->AddSeparatorItem();
		//Close window
			menu->AddItem( new BMenuItem("Close",new BMessage(B_QUIT_REQUESTED),'W'));
		//Quit
			item = new BMenuItem("Quit",new BMessage(B_QUIT_REQUESTED),'Q');
			item->SetTarget(be_app);
			menu->AddItem(item);
	menu_bar->AddItem(menu);
	return menu_bar;
}


GenericWindow::~GenericWindow()
{ }


bool GenericWindow::Save()
{
	const char* text = m_text_view->Text();
	int32 length = m_text_view->TextLength();
	const char* filename = GetFileName();

	BFile save_file(filename,B_WRITE_ONLY|B_CREATE_FILE|B_ERASE_FILE);
	if(save_file.InitCheck() != B_NO_ERROR)
		return false;
	if(save_file.Write(text,length) != length)
		return false;
	BNodeInfo node_info(&save_file);
	node_info.SetType("text/plain");
	return true;
}


//******************************************************************************************************
//**** TEXT VIEW
//******************************************************************************************************
GenericTextView::GenericTextView(BRect a_frame,const char* a_name,int32 a_resize_mode,int32 a_flags)
: WrappingTextView(a_frame,a_name,a_resize_mode,a_flags)
{ }


void GenericTextView::Modified()
{
	FileDocumentWindow* parent = (FileDocumentWindow*)Window();
	if(parent)
		parent->SetModifiedFlag();
}

